function m = createBvModel(weather, startTime, controls, indoor)
%CREATEBVMODEL Create a StateSpaceModel object based on the Vanthoor model, with the addition of lamps and growpipes
% Based on:
%   [1] Vanthoor, B., Stanghellini, C., van Henten, E. J. & de Visser, P. H. B. 
%       A methodology for model-based greenhouse design: Part 1, a greenhouse climate 
%       model for a broad range of designs and climates. Biosyst. Eng. 110, 363-377 (2011).
%   [2] Vanthoor, B., de Visser, P. H. B., Stanghellini, C. & van Henten, E. J. 
%       A methodology for model-based greenhouse design: Part 2, description and 
%       validation of a tomato yield model. Biosyst. Eng. 110, 378-395 (2011).
% These are also available in
%   [3] Vanthoor, B. A model based greenhouse design method. (Wageningen University, 2011).
%
% Function inputs:
%   weather         A matrix with 8 columns, in the following format:
%       weather(:,1)    timestamps of the input [s] in regular intervals
%       weather(:,2)    radiation     [W m^{-2}]  outdoor global irradiation 
%       weather(:,3)    temperature   [C]        outdoor air temperature
%       weather(:,4)    humidity      [kg m^{-3}] outdoor vapor concentration
%       weather(:,5)    co2 [kg{CO2} m^{-3}{air}] outdoor CO2 concentration
%       weather(:,6)    wind        [m s^{-1}] outdoor wind speed
%       weather(:,7)    sky temperature [C]
%       weather(:,8)    temperature of external soil layer [C]
%
%   startTime       date and time of starting point (datetime)
%
%   controls        (optional) A matrix with 8 columns, in the following format:
%       controls(:,1)     timestamps of the input [s] in regular intervals of 300, starting with 0
%       controls(:,2)     Energy screen closure 			0-1 (1 is fully closed)
%       controls(:,3)     Black out screen closure			0-1 (1 is fully closed)
%       controls(:,4)     Average ventilation aperture		0-1 (1 is fully open)
%       controls(:,5)     Pipe rail temperature 			C
%       controls(:,6)     Grow pipes temperature 			C
%       controls(:,7)     Toplights on/off                  0/1 (1 is on)
%       controls(:,8)     Interlight on/off                 0/1 (1 is on)
%       controls(:,9)     CO2 injection                     0/1 (1 is on)
%
%   indoor          (optional) A 3 column matrix with:
%       indoor(:,1)     timestamps of the input [s] in regular intervals of 300, starting with 0
%       indoor(:,2)     temperature       [C]             indoor air temperature
%       indoor(:,3)     vapor pressure    [Pa]             indoor vapor concentration
%       indoor(:,4)     co2 concentration [mg m^{-3}]      indoor vapor concentration

% David Katzin, Wageningen University
% david.katzin@wur.nl

    if ~exist('indoor','var')
        indoor = [];
    end

    m = StateSpaceModel();
    setBvParams(m); % define parameters and nominal values
    m.d = setBvInput(weather);  % define and set inputs
    m.t = setBvTime(m, startTime); % define time phase
    m.u = setBvControls(); % define controls
    m.x = setBvStates(); % define states 
    
    if exist('controls','var') && ~isempty(controls)
        ruleBased = false;
        
        % add control trajectories
        time = controls(:,1);
        m.u.thScr = DynamicElement('u.thScr', [time controls(:,2)]);
        
        m.u.blScr = DynamicElement('u.blScr', [time controls(:,3)]); 
        
        m.u.roof = DynamicElement('u.roof', [time controls(:,4)]);
        m.d.tPipe = DynamicElement('d.tPipe', [time controls(:,5)]);
        m.d.tGroPipe = DynamicElement('d.tGroPipe', [time controls(:,6)]);
        
        % is 1 if pipe is about to be switched off
        m.d.pipeSwitchOff = DynamicElement('d.pipeSwitchOff', ...
            [time ((controls(:,5)~=0) & (circshift(controls(:,5),-1)==0))]);
        
        % is 1 if grow pipe is about to be switched off
        m.d.groPipeSwitchOff = DynamicElement('d.groPipeSwitchOff', ...
            [time ((controls(:,6)~=0) & (circshift(controls(:,6),-1)==0))]);
        
        m.u.lamp = DynamicElement('u.lamp', [time controls(:,7)]);
        
        m.u.extCo2 = DynamicElement('u.extCo2', [time controls(:,9)]);
        
        m.u.intLamp = DynamicElement('u.intLamp', [time controls(:,8)]);

        % controls not considered
        m.u.boil = DynamicElement('u.boil', [time zeros(size(time))]);
        m.u.shScrPer = DynamicElement('u.shScrPer', [time zeros(size(time))]);
        m.u.side = DynamicElement('u.side', [time zeros(size(time))]);
        m.u.groBoil = DynamicElement('u.groBoil', [time zeros(size(time))]);
        m.u.shScr = DynamicElement('u.shScr', [time zeros(size(time))]);
    else
        ruleBased = true;
    end
        
    setBvAux(m); % define auxiliary states
    
    if ruleBased
        m.u = setBvControlRules(m); % define control rules
    end
    
    setBvOdes(m); % define odes - must be done after the aux states and control rules are set
    m.x = setBvInit(m, indoor); % set initial values for the states
end